/* Pre-TD — League-only, PNG sequence, per-team render
   Project: ScoringAnimations2.aep
   Comp:    Pre-TD
*/

(function () {
  // --- load common (force reload) ---
  (function(){
    var COMMON = $.getenv("AE_COMMON_JSX") || "";
    if (COMMON) { try { delete $.global.GL; } catch(e) { $.global.GL = undefined; } $.evalFile(File(COMMON)); }
    if (!$.global.GL) throw new Error("Common library not loaded. Set AE_COMMON_JSX to gl_common.jsxinc");
  })();

  var GL = $.global.GL;

  // --- env ---
  var PROJECT   = GL.env("AE_PROJECT", null);
  var CSV_PATH  = GL.env("AE_CSV", null);
  var COMP_NAME = GL.env("AE_COMP","Pre-TD");

  var TOUCHDOWN_LAYER          = GL.env("AE_TOUCHDOWN_LAYER","TOUCHDOWN");
  var TEAMNAME_LAYER           = GL.env("AE_TEAMNAME_LAYER","TeamName");
  var LOGO_LAYER               = GL.env("AE_LOGO_LAYER","Logo");

  // These are layer-name **prefixes** in the main comp
  var MAIN_LOGO_PRECOMP_LAYER  = GL.env("AE_MAIN_LOGO_PRECOMP_LAYER","Main-Logo-PreComp");
  var LOGO_PRECOMP_LAYER_NAME  = GL.env("AE_LOGO_PRECOMP_LAYER_NAME","Logo PreComp");

  var LEAGUE    = GL.env("AE_LEAGUE","NFL");
  var LIMIT_STR = GL.env("AE_LIMIT","");
  var LIMIT     = (LIMIT_STR && !isNaN(parseInt(LIMIT_STR,10))) ? parseInt(LIMIT_STR,10) : null;

  var OUTDIR    = GL.env("AE_OUTDIR","");
  var PATH_TPL  = GL.env("AE_PATH_TEMPLATE","{league}");
  var ANIM_NAME = GL.env("AE_ANIM","_PreTD");
  var RS_TPL    = GL.env("AE_RS_TEMPLATE","Best Settings");
  var OM_TPL    = GL.env("AE_OM_TEMPLATE","PNG Sequence");
  var PURGE     = (GL.env("AE_PURGE_BEFORE_RENDER","1")==="1");
  var NO_RENDER = (GL.env("AE_NO_RENDER","0")==="1");
  var QUIT_APP  = (GL.env("AE_QUIT","1")==="1");

  var PIXEL_SNAP= (GL.env("AE_PIXEL_SNAP","1")==="1");

  // --- start ---
  if (app.beginSuppressDialogs){ try{ app.beginSuppressDialogs(); }catch(e){} }
  app.beginUndoGroup("Pre-TD - League Only - PNG Seq");

  if(!PROJECT) GL.fail("AE_PROJECT env not set.");
  var aep=new File(PROJECT); if(!aep.exists) GL.fail("AE_PROJECT not found: "+PROJECT);
  app.open(aep);

  if(!CSV_PATH) GL.fail("AE_CSV env not set.");
  if(!LEAGUE || GL.cleanValue(LEAGUE)==="") GL.fail("AE_LEAGUE is required.");

  // --- CSV with team_id ---
  var rows = GL.parseCSV(GL.openRead(CSV_PATH));
  var H = rows[0], M = {}; for (var i=0;i<H.length;i++) M[GL.toLower(H[i])] = i;
  function need(k){ if(M[k]===undefined) GL.fail("Missing column: "+k); }
  need("abbreviation"); need("r"); need("g"); need("b"); need("r2"); need("g2"); need("b2");
  function opt(){ for (var i=0;i<arguments.length;i++){ var k=arguments[i]; if (M[k]!==undefined) return M[k]; } return undefined; }

  var IDX = {
    league:  opt("league"),
    conf:    opt("conference","conf","group"),
    name:    opt("displayname","name"),
    team_id: opt("team_id")
  };

  function buildTeams(){
    var out=[], leagueDefault=LEAGUE;
    for (var r=1; r<rows.length; r++){
      var row = rows[r]; if (!row || !row.length) continue;
      var ab = GL.cleanValue(row[M["abbreviation"]]); if(!ab) continue;
      out.push({
        abbr: ab,
        league: (IDX.league!==undefined ? GL.cleanValue(row[IDX.league]) : GL.cleanValue(leagueDefault||"NA")),
        conference: (IDX.conf!==undefined ? GL.cleanValue(row[IDX.conf]) : ""),
        name: (IDX.name!==undefined ? GL.cleanValue(row[IDX.name]) : ab),
        team_id: (IDX.team_id!==undefined ? GL.cleanValue(row[IDX.team_id]) : ""),
        primary:   GL.rgb01(row[M["r"]],  row[M["g"]],  row[M["b"]]),
        secondary: GL.rgb01(row[M["r2"]], row[M["g2"]], row[M["b2"]])
      });
    }
    return out;
  }

  var teamsAll = buildTeams();
  var todo = (function(all, leagueStr){
    var res=[], targetBase = GL.normalizeLeagueBase(leagueStr);
    for (var i=0;i<all.length;i++){
      var teamBase = GL.normalizeLeagueBase(all[i].league);
      if (teamBase === targetBase) res.push(all[i]);
    }
    return res;
  })(teamsAll, LEAGUE);

  if (LIMIT && todo.length>LIMIT) todo = todo.slice(0, LIMIT);
  if(!todo.length) GL.fail("No teams matched league: "+LEAGUE);

  var comp = GL.findComp(COMP_NAME);
  if(!comp) GL.fail("Comp not found: "+COMP_NAME);

  var rootOut = OUTDIR ? new Folder(OUTDIR) : (app.project.file ? app.project.file.parent : Folder.desktop);
  GL.ensureFolder(rootOut);

  var logoOpts = {
    dir: GL.env("AE_LOGO_DIR",""),
    tpl: GL.env("AE_LOGO_PATH_TEMPLATE","{league}/{abbr}"),
    exts: GL.env("AE_LOGO_EXTS","png,jpg,jpeg,svg,ai,psd")
  };

  // ---------- Helpers ----------

  function setTouchdownFillPrimary(compItem, color01){
    var ly = GL.getLayer(compItem, TOUCHDOWN_LAYER);
    if (!ly) return false;
    return GL.setTextOrFillColor(ly, color01); // disables stroke (desired)
  }

  function setTeamNameStrokePrimary(compItem, teamName, color01){
    var ly = GL.getLayer(compItem, TEAMNAME_LAYER);
    if (!ly) return false;
    GL.setTextContent(ly, teamName);
    try{
      var st = ly.property("Source Text");
      if (st){
        var td = st.value;
        td.applyFill = false;
        td.applyStroke = true;
        td.strokeColor = GL.safeColor(color01);
        td.strokeWidth = 2;
        td.strokeOverFill = true;
        st.setValue(td);
        return true;
      }
    }catch(e){}
    // Layer Style fallback
    try{
      var styles = ly.property("ADBE Layer Styles") || ly.property("Layer Styles");
      if (!styles) styles = ly.addProperty("ADBE Layer Styles");
      var s = styles.property("ADBE Stroke") || styles.property("Stroke");
      if (!s){ try{ s = styles.addProperty("ADBE Stroke"); }catch(e){ s=null; } }
      if (s){
        var en = s.property("ADBE Layer Styles-0001") || s.property("Enabled");
        var c  = s.property("ADBE Stroke-0002") || s.property("Color");
        var op = s.property("ADBE Stroke-0003") || s.property("Opacity");
        var sz = s.property("ADBE Stroke-0007") || s.property("Size");
        if (en) try{ en.setValue(true); }catch(e){}
        if (c)  try{ c.setValue(GL.safeColor(color01)); }catch(e){}
        if (op) try{ op.setValue(100); }catch(e){}
        if (sz) try{ sz.setValue(2); }catch(e){}
        return true;
      }
    }catch(e){}
    return false;
  }

  // Import (cache) team logo footage once
  var __logoCache = {};
  function importTeamLogoFootage(team, opts){
    var f = GL.findLogoFile(team.league, team.abbr, opts);
    if (!f) return null;
    var key = f.fsName;
    if (__logoCache[key]) return __logoCache[key];
    try{
      var io = new ImportOptions(f);
      if (!io.canImportAs(ImportAsType.FOOTAGE)) return null;
      var footage = app.project.importFile(io);
      __logoCache[key] = footage;
      return footage;
    }catch(e){ return null; }
  }

  // Replace the single main "Logo" layer
  function replaceLogoOnLayer(compItem, layerName, team, opts){
    var ly = GL.getLayer(compItem, layerName);
    if (!ly) return false;
    var footage = importTeamLogoFootage(team, opts);
    if (!footage) return false;
    try { ly.replaceSource(footage, false); return true; } catch(e){ return false; }
  }

  // ---------- NEW: Deep replace helpers ----------

  // prefix matcher for main-comp precomp layers (handles "Name", "Name 2", "Name (copy)", etc.)
  function hasPrefix(name, base){
    var esc = String(base||"").replace(/[.*+?^${}()|[\]\\]/g, "\\$&");
    var re  = new RegExp("^"+esc+"(?:$|[\\s_()-])","i");
    return re.test(String(name||""));
  }

  // Replace every layer named TeamLogo* in a comp, recursing into nested precomps
  function replaceTeamLogoDeep(inComp, footage, depth){
    if (!inComp || !(inComp instanceof CompItem)) return 0;
    if (depth > 6) return 0; // safety
    var changed = 0, reTL = /^TeamLogo(?:[\s_()-]*\d*)?$/i;

    for (var i=1; i<=inComp.numLayers; i++){
      var L = inComp.layer(i);
      var nm = String(L && L.name || "");
      // If this is a TeamLogo* layer, replace by reference (works regardless of prior source type)
      if (reTL.test(nm)){
        try { L.replaceSource(footage, false); changed++; } catch(e){}
      }
      // Recurse into nested precomps
      try{
        if (L && L.source && (L.source instanceof CompItem)){
          changed += replaceTeamLogoDeep(L.source, footage, depth+1);
        }
      }catch(e){}
    }
    return changed;
  }

  // For each layer in the main comp whose name starts with baseNamePrefix, dive in and replace TeamLogo* deeply
  function replaceInsideAllMatchingPrecomps(mainComp, baseNamePrefix, team, opts){
    var total = 0;
    var footage = importTeamLogoFootage(team, opts);
    if (!footage) return 0;

    for (var i=1; i<=mainComp.numLayers; i++){
      var L = mainComp.layer(i);
      if (!(L && L.source && (L.source instanceof CompItem))) continue;
      if (!hasPrefix(L.name, baseNamePrefix)) continue;
      total += replaceTeamLogoDeep(L.source, footage, 0);
    }
    return total;
  }

  // Safe pixel snap on 2D/non-collapsed only
  function applyPixelSnapIfSafe(layer){
    if (!layer) return false;
    try { if (layer.threeDLayer === true || layer.collapseTransformation === true) return false; } catch(e){}
    var p = layer.property("Position");
    if (!p || !p.canSetExpression) return false;
    var expr = "var v = value; [Math.round(v[0]), Math.round(v[1])];";
    try { p.expression = expr; return true; } catch(e){ return false; }
  }

  // --- RQ ---
  GL.rqClear();

  // --- per team ---
  for (var i=0;i<todo.length;i++){
    var t = todo[i];

    var smart = GL.computeSmartColors( GL.safeColor(t.primary), GL.safeColor(t.secondary) );
    var smartP = smart.primary;

    // 1) TOUCHDOWN fill = primary
    setTouchdownFillPrimary(comp, smartP);

    // 2) TeamName stroke-only = primary (2px)
    setTeamNameStrokePrimary(comp, t.abbr, smartP);

    // 3) Main "Logo" layer
    replaceLogoOnLayer(comp, LOGO_LAYER, t, logoOpts);

    // 4) Deep-replace ALL TeamLogo* inside any "Main-Logo-PreComp*" in the main comp
    replaceInsideAllMatchingPrecomps(comp, MAIN_LOGO_PRECOMP_LAYER, t, logoOpts);

    // 5) Deep-replace ALL TeamLogo* inside any "Logo PreComp*" in the main comp
    replaceInsideAllMatchingPrecomps(comp, LOGO_PRECOMP_LAYER_NAME, t, logoOpts);

    // 6) Optional pixel snap on safe targets
    if (PIXEL_SNAP){
      applyPixelSnapIfSafe( GL.getLayer(comp, TOUCHDOWN_LAYER) );
      applyPixelSnapIfSafe( GL.getLayer(comp, TEAMNAME_LAYER) );
      applyPixelSnapIfSafe( GL.getLayer(comp, LOGO_LAYER) );
      // we intentionally do NOT snap the precomp container layers
    }

    if (PURGE && app.purge) { try{ app.purge(PurgeTarget.ALL_CACHES); }catch(e){} }

    if (!NO_RENDER){
      var lc = GL.leagueAndConfForPath(t.league, t.conference);
      // CFB: use team_id if present
      if (lc.base === "CFB" && t.team_id && String(t.team_id)!=="") lc.conf = String(t.team_id);
      var paths = GL.outPaths(rootOut, PATH_TPL, lc.base, t.abbr, ANIM_NAME, lc.conf);
      GL.rqRenderTo(comp, RS_TPL, OM_TPL, paths.file);
    }
  }

  app.endUndoGroup();
  if (app.endSuppressDialogs){ try{ app.endSuppressDialogs(); }catch(e){} }
  if (QUIT_APP) app.quit();
})();
